﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO.Ports;

namespace EasyCAT_USB_CS
{
    public partial class MainWindow : Form
    {
        const int TRUE = 1;
        const int FALSE = 0;

        const int END = 0xC0;
        const int ESC = 0xDB;
        const int ESC_END = 0xDC;
        const int ESC_ESC = 0xDD;

        byte[] InBuff = new byte[32];
        byte[] OutBuff = new byte[32 + 1 + 1];

        byte[] SlipTxBuff = new byte[128];
        byte[] SlipRxBuff = new byte[128];

        bool m_bUpdateInputs = true;

        byte[] ReceiveBuffer = new byte[128];
        int m_iCountBuffer = 0;

        delegate void SerialEventsCallback();

        public MainWindow()
        {
            InitializeComponent();
        }

        private void btnStart_Click(object sender, EventArgs e)
        {
            if (comboBoxSerialPorts.SelectedItem != null && comboBoxSerialPorts.SelectedItem.ToString() != "")
            {
                try
                {
                    //configuring the serial port
                    serialPortUSB.PortName = comboBoxSerialPorts.SelectedItem.ToString();
                    serialPortUSB.BaudRate = 115200;
                    serialPortUSB.DataBits = 8;
                    serialPortUSB.Parity = Parity.None;
                    serialPortUSB.StopBits = StopBits.One;

                    //opening the serial port
                    serialPortUSB.Open();

                    timerSend.Interval = 100;
                    timerSend.Enabled = true;
                }
                catch (Exception ex)
                {
                    MessageBox.Show(ex.Message,
                     "EasyCAT USB",
                     MessageBoxButtons.OK,
                     MessageBoxIcon.Exclamation);
                }
            }
        }

        private void btnStop_Click(object sender, EventArgs e)
        {
            timerSend.Enabled = false;

            //close the port
            if (serialPortUSB.IsOpen)
            {
                serialPortUSB.Close();
            }

            lbl_USBStatus.ForeColor = Color.Blue;
            lbl_EthStatus.ForeColor = Color.Blue;

            lbl_USBStatus.Text = "Closed";
            lbl_EthStatus.Text = "Not Operational";
        }

        private void timerSend_Tick(object sender, EventArgs e)
        {
            if (serialPortUSB.IsOpen)
            {
                TxSlipFrame();                                              //

                InBuff[30] -= 1;                                            // generazione di due denti di sega
                InBuff[31] += 1;                                            // negli ultimi byte per test
            }
        }

        private void SerialEvent()
        {
            // Obtain the number of bytes waiting in the port's buffer
            int bytes = serialPortUSB.BytesToRead;

            // Create a byte array buffer to hold the incoming data
            byte[] buffer = new byte[bytes];

            // Read the data from the port and store it in our buffer
            serialPortUSB.Read(buffer, 0, bytes);

            foreach (byte b in buffer)
            {
                if (m_iCountBuffer >= 128)
                {
                    m_iCountBuffer = 0;
                }

                ReceiveBuffer[m_iCountBuffer] = b;
                m_iCountBuffer++;
            }

            int NumChar = m_iCountBuffer;                          // lettura dati da linea seriale

            if (NumChar < 36)                                           // se non si sono ricevuti almeno
            {                                                           // 36 caratteri il frame non e' valido
                /*WatchDog();

                if (lbl_USBStatus.Text != "Error RX")
                {
                    lbl_USBStatus.ForeColor = Color.Red;
                    lbl_USBStatus.Text = "Error RX";
                }*/
            }
            else                                                        // perche' il frame sia valido la funzione di unwrap
            {                                                           // deve restituire "True" ed il byte di stato                                                               // deve segnalare "operational" (0x08)
                for (int i = 0; i < NumChar && i < 128; i++)
                    SlipRxBuff[i] = ReceiveBuffer[i];

                m_iCountBuffer = 0;

                if ((UnWrapSlipFrame(ref SlipRxBuff, ref OutBuff) == TRUE) /*&& (OutBuff[32] == 0x08)*/)
                {
                    if (OutBuff[32] == 0x08)
                    {
                        if (lbl_EthStatus.Text != "Operational")
                        {
                            lbl_EthStatus.ForeColor = Color.Green;
                            lbl_EthStatus.Text = "Operational";
                        }
                    }
                    else
                    {
                        WatchDog();
                    }

                    if (lbl_USBStatus.Text != "Running")
                    {
                        lbl_USBStatus.ForeColor = Color.Green;
                        lbl_USBStatus.Text = "Running";
                    }

                    UpdateInputs();
                    UpdateOutputs();
                }
                else
                {
                    WatchDog();

                    if (lbl_USBStatus.Text != "Error RX 2")
                    {
                        lbl_USBStatus.ForeColor = Color.Red;
                        lbl_USBStatus.Text = "Error RX 2";
                    }
                }
            }
        }

        private void serialPortUSB_DataReceived(object sender, System.IO.Ports.SerialDataReceivedEventArgs e)
        {
            SerialEventsCallback d = new SerialEventsCallback(SerialEvent);
            //this.Invoke(d, new object[] { });
            this.BeginInvoke(d, new object[] { });
        }

        private void MainWindow_FormClosed(object sender, FormClosedEventArgs e)
        {
            //close the port
            if (serialPortUSB.IsOpen)
            {
                serialPortUSB.Close();
            }
        }

        private void MainWindow_Load(object sender, EventArgs e)
        {
            // Get a list of serial port names.
            string[] ports = SerialPort.GetPortNames();

            // Display each port name to the console.
            foreach (string port in ports)
            {
                comboBoxSerialPorts.Items.Add(port);
            }

            string sStringa;

            int iRow;

            for (int i = 0; i < 32; i++)
            {
                sStringa = "Byte ";
                sStringa += i.ToString();
                iRow = Grid_Table.Rows.Add(new object[] { sStringa, 0, 0 });

                Grid_Table.Rows[iRow].Cells[1].Style.BackColor = Color.Lime;
                Grid_Table.Rows[iRow].Cells[2].Style.BackColor = Color.Yellow;
                Grid_Table.Rows[iRow].Cells[2].Style.SelectionBackColor = Color.Yellow;
                Grid_Table.Rows[iRow].Cells[1].Style.Format = "X2";
                Grid_Table.Rows[iRow].Cells[2].Style.Format = "X2";
            }
        }

        void TxSlipFrame()
        {
            int i;
            int j;

            j = 0;

            SlipTxBuff[j] = END;
            j++;

            for (i = 0; i < 32; i++)
            {
                switch (InBuff[i])
                {
                    case END:
                        SlipTxBuff[j] = ESC;
                        j++;

                        SlipTxBuff[j] = ESC_END;
                        j++;
                        break;

                    case ESC:
                        SlipTxBuff[j] = ESC;
                        j++;
                        SlipTxBuff[j] = ESC_ESC;
                        j++;
                        break;

                    default:
                        SlipTxBuff[j] = InBuff[i];
                        j++;
                        break;
                }
            }

            SlipTxBuff[j] = END;
            j++;

            serialPortUSB.ReadExisting();

            //write data to serial port
            serialPortUSB.Write(SlipTxBuff, 0, j);
        }

        byte UnWrapSlipFrame(ref byte[] BuffIn, ref byte[] BuffOut)
        {
            int i;
            byte Result = FALSE;
            byte FrameComplete = FALSE;
            byte PreviousRxChar = 0x00;
            byte RxChar;
            byte Len = 0;


            if (BuffIn[0] != END)                            // verifica che il frame inizi con END
            {                                               //
                Result = FALSE;                             //
                return Result;                              //
            }                                               //
            else                                            //----loop di analisi del buffer di ingresso ----
            {
                for (i = 1; i < 128; i++)
                {
                    RxChar = BuffIn[i];

                    switch (RxChar)                          // analizza il carattere ricevuto
                    {
                        case ESC:                           //--- e' ESC: si aspetta il carattere successivo ---
                            PreviousRxChar = RxChar;            //
                            break;                              //

                        case END:                           //--- e' END: il frame e' finito -------------------
                            if (Len == 32 + 1 + 1)                      // se si sono ricevuti 33 caratteri il frame e' valido
                                Result = TRUE;                  //
                            else                                //
                                Result = FALSE;                 //
                            //
                            FrameComplete = TRUE;               // in ogni caso esci
                            break;                              //

                        default:                            //--- e' un carattere normale ----------------------
                            //
                            if (PreviousRxChar == ESC)           // se il carattere precedente era ESC bisogna
                            {                                   // decodificare il carattere ricevuto
                                PreviousRxChar = RxChar;        //
                                //
                                switch (RxChar)                  //
                                {                               //
                                    case ESC_END:               // e' un carattere uguale a END
                                        RxChar = END;               //
                                        break;                      //
                                    //
                                    case ESC_ESC:               // e' un carattere uguale a ESC
                                        RxChar = ESC;               //
                                        break;                      //
                                }                               //
                            }                                   //

                            else                                // si usa il carattere senza modificarlo
                            {                                   //
                                PreviousRxChar = RxChar;        //
                            }                                   //

                            BuffOut[Len] = RxChar;              // metti il carattere ricevuto nel buffer
                            Len++;                              // incrementa il numero di caratteri ricevuti

                            if (Len > 32 + 1 + 1)                        // se c'e' buffer overflow esci segnalando
                            {                                   // frame non valido
                                Result = FALSE;                 //
                                FrameComplete = TRUE;           //
                            }                                   //
                            break;                              //
                    }

                    if (FrameComplete == TRUE)              // il frame e' terminato
                        break;                              // esci dal loop
                }
            }
            return Result;
        }

        void WatchDog()                                        // gestione watchdog
        {
            int i;

            if (lbl_EthStatus.Text != "Not Operational")
            {
                lbl_EthStatus.ForeColor = Color.Red;
                lbl_EthStatus.Text = "Not Operational";
            }

            for (i = 0; i < 32; i++)                                    // azzera il buffer degli output
            {                                                       //
                OutBuff[i] = 0x00;                                  //
            }                                                       //
        }

        void UpdateInputs()
        {
            if (!m_bUpdateInputs)
                return;

            for (int i = 0; i < 32; i++)
            {
                Grid_Table.Rows[i].Cells[1].Value = InBuff[i];
            }
        }

        void UpdateOutputs()
        {
            for (int i = 0; i < 32; i++)
            {
                Grid_Table.Rows[i].Cells[2].Value = OutBuff[i];
            }
        }

        private void Grid_Table_CellEndEdit(object sender, DataGridViewCellEventArgs e)
        {
            if (e.RowIndex > -1 && e.ColumnIndex == 1)
            {
                int intValue = int.Parse(Grid_Table.Rows[e.RowIndex].Cells[e.ColumnIndex].Value.ToString(), System.Globalization.NumberStyles.HexNumber);

                InBuff[e.RowIndex] = (byte)intValue;

                m_bUpdateInputs = true;
            }
        }

        private void Grid_Table_CellBeginEdit(object sender, DataGridViewCellCancelEventArgs e)
        {
            if (e.RowIndex > -1 && e.ColumnIndex == 1)
            {
                m_bUpdateInputs = false;
            }
        }
    }
}
